package com.ruoyi.web.controller.grid;

import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.system.domain.grid.enums.GridRegionSearchParam;
import com.ruoyi.system.mapper.grid.GridRegionMapper;
import com.ruoyi.system.service.grid.*;
import io.swagger.annotations.Api;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.util.*;

/**
 * @author : peter
 * @description: 网格对象采集
 * @date : 2022-05-04 18:04
 */
@Api(tags = "网格管理 - 网格对象采集")
@RestController
@RequestMapping("/api/grid/object")
public class GridObjectController {

    @Autowired
    private GridPeopleService gridPeopleService;
    @Autowired
    private GridHouseService gridHouseService;
    @Autowired
    private GridCompanyService gridCompanyService;
    @Autowired
    private GridShopService gridShopService;
    @Autowired
    private GridRegionMapper gridRegionMapper;

    /**
     * 网格对象按字典分类型统计和总量
     *
     * @param wgCode
     * @return
     */
    @GetMapping(value = "/infoWithType")
    public AjaxResult infoWithType(@RequestParam(value = "wgCode", required = false) String wgCode) {
        Map<String, Object> map = new HashMap<>();

        // 人口
        List<Map<String, Object>> peopleList = gridPeopleService.countGridPeople(wgCode);
        map.put("people", peopleList);
        map.put("peopleAllNum", gridPeopleService.countAll(wgCode));
        map.put("peopleRkbq", gridPeopleService.countGridPeopleByRkbq(wgCode));
        map.put("specialPeopleNum", gridPeopleService.countSpecialPeople(wgCode) != null ? gridPeopleService.countSpecialPeople(wgCode) : 0);

        // 住宅类型分类统计
        int houseAllNum = gridHouseService.countAll(wgCode);
        List<Map<String, Object>> houseList = gridHouseService.countGridHouse(wgCode);
        map.put("house", houseList);
        map.put("houseAllNum", houseAllNum);

        // 住宅状态分类统计
        List<Map<String, Object>> houseFwztList = gridHouseService.countGridHouseByFwzt(wgCode);
        map.put("houseFwzt", houseFwztList);
        BigDecimal houseAllCount = new BigDecimal(houseAllNum);
        Iterator<Map<String, Object>> iterator = houseFwztList.iterator();
        while (iterator.hasNext()) {
            Map<String, Object> mapH = iterator.next();
            if (houseAllNum != 0) {
                BigDecimal percent = new BigDecimal(String.valueOf(mapH.get("number"))).divide(houseAllCount, 3, BigDecimal.ROUND_HALF_UP).stripTrailingZeros();
                NumberFormat perFormat = NumberFormat.getPercentInstance(Locale.CHINA);
                perFormat.setMinimumFractionDigits(0);
                mapH.put("percent", perFormat.format(percent));
            } else {
                mapH.put("percent", "0%");
            }
        }

        // 企业
        List<Map<String, Object>> companyList = gridCompanyService.countGridCompany(wgCode);
        map.put("company", companyList);
        map.put("companyAllNum", gridCompanyService.countAll(wgCode));

        // 商铺
        List<Map<String, Object>> shopList = gridShopService.countGridShop(wgCode);
        map.put("shop", shopList);
        map.put("shopAllNum", gridShopService.countAll(wgCode));

        // 地块
        GridRegionSearchParam param = new GridRegionSearchParam();
        param.setLevel("4");
        if (StringUtils.isNotBlank(wgCode)) {
            List<String> wgCodes = Arrays.asList(wgCode.split(","));
            param.setWgCodeLikes(wgCodes);
        }
        int wgCount = gridRegionMapper.countByParams(param);
        map.put("wgCount", wgCount);
        return AjaxResult.success(map);
    }

    /**
     * 网格对象总量统计
     *
     * @param wgCode
     * @return
     */
    @GetMapping(value = "/info")
    public AjaxResult info(@RequestParam(value = "wgCode", required = false) String wgCode) {
        Map<String, Object> map = new HashMap<>();
        map.put("peopleAllNum", gridPeopleService.countAll(wgCode));
        map.put("houseAllNum", gridHouseService.countAll(wgCode));
        map.put("companyAllNum", gridCompanyService.countAll(wgCode));
        map.put("shopAllNum", gridShopService.countAll(wgCode));
        return AjaxResult.success(map);
    }

    /**
     * 大屏上村社区基本信息接口
     *
     * @param wgCode
     * @return
     */
    @GetMapping(value = "/basicInfo")
    public AjaxResult basicInfo(@RequestParam(value = "wgCode", required = false) String wgCode) {
        List<Map<String, Object>> list = new ArrayList<>();
        Map<String, Object> map = new HashMap<>();
        map.put("name", "人口总数");
        map.put("number", gridPeopleService.countAll(wgCode));
        list.add(map);

        Map<String, Object> map1 = new HashMap<>();
        map1.put("name", "住宅总数");
        map1.put("number", gridHouseService.countAll(wgCode));
        list.add(map1);

        List<Map<String, Object>> peopleList = gridPeopleService.countGridPeople(wgCode);
        Map<String, Object> map2 = new HashMap<>();
        map2.put("name", "户籍人口");
        map2.put("number", peopleList.get(0).get("number"));
        list.add(map2);

        Map<String, Object> map3 = new HashMap<>();
        map3.put("name", "非户籍人口");
        map3.put("number", peopleList.get(1).get("number"));
        list.add(map3);

        Map<String, Object> map4 = new HashMap<>();
        map4.put("name", "特殊人口");
        map4.put("number", gridPeopleService.countSpecialPeople(wgCode) != null ? gridPeopleService.countSpecialPeople(wgCode) : 0);
        list.add(map4);
        return AjaxResult.success(list);
    }

    /**
     * 大屏上二级/三级网格人口类型数量
     */
    @GetMapping(value = "/countWsPeopleRklxByGrid")
    public AjaxResult countWsPeopleRklxByGrid(@RequestParam(value = "wgCode", required = false) String wgCode) {
        return AjaxResult.success(gridPeopleService.countGridPeopleRklxByGrid(wgCode));
    }

    /**
     * 各村社区网格对象数量统计
     */
    @GetMapping(value = "/infoWithGrid")
    public AjaxResult infoWithGrid() {
        Map<String, Object> map = new HashMap<>();
        map.put("peopleGrid", gridPeopleService.countGridPeopleByGrid());
        map.put("houseGrid", gridHouseService.countGridHouseByGrid());
        map.put("companyGrid", gridCompanyService.countGridCompanyByGrid());
        map.put("shopGrid", gridShopService.countGridShopByGrid());
        return AjaxResult.success(map);
    }

}
