package com.ruoyi.web.controller.party;

import com.google.common.collect.Lists;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.domain.entity.SysDictData;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.party.domain.PartyActivity;
import com.ruoyi.party.domain.vo.PartyActivityVO;
import com.ruoyi.party.service.api.IPartyActivityService;
import com.ruoyi.system.mapper.system.SysDictDataMapper;
import io.swagger.annotations.Api;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 党员活动Controller
 * 
 * @author ruoyi
 * @date 2024-05-15
 */
@Api(tags = "党建-党员活动")
@RestController
@RequestMapping("/api/party/partyActivity")
public class PartyActivityController extends BaseController
{
    @Autowired
    private IPartyActivityService partyActivityService;

    @Autowired
    private SysDictDataMapper sysDictDataMapper;

    private List<SysDictData> getDataByType(){
        return sysDictDataMapper.selectDictDataByType("service_type");
    }

    /**
     * 查询党员活动列表
     */
    @GetMapping("/selectByInfo")
    public TableDataInfo list(PartyActivity partyActivity)
    {
        startPage();
        List<PartyActivity> list = partyActivityService.selectPartyActivityList(partyActivity);
        TableDataInfo dataInfo = getDataTable(list);
        if(!CollectionUtils.isEmpty(dataInfo.getRows())){
            dataInfo.setRows(partyActivityService.convertData((List<PartyActivity>) dataInfo.getRows(), getDataByType()));
        }
        return dataInfo;
    }

    /**
     * 导出党员活动列表
     */
    @Log(title = "党员活动", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, PartyActivity partyActivity)
    {
        List<PartyActivity> list = partyActivityService.selectPartyActivityList(partyActivity);
        if(!CollectionUtils.isEmpty(list)){
            list = partyActivityService.convertData(list, getDataByType());
        }
        ExcelUtil<PartyActivity> util = new ExcelUtil<PartyActivity>(PartyActivity.class);
        util.exportExcel(response, list, "党员活动数据");
    }

    /**
     * 获取党员活动详细信息
     */
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") String id)
    {
        List<SysDictData> dictDataList = this.getDataByType();
        Map<String, SysDictData>  map = dictDataList.stream().collect(Collectors.toMap(SysDictData::getDictValue, x -> x));
        PartyActivityVO partyActivityVO = partyActivityService.selectPartyActivityById(id);
        if(StringUtils.isNotBlank(partyActivityVO.getFwlx())){
            SysDictData sysDictData = map.get(partyActivityVO.getFwlx());
            if(sysDictData != null){
                partyActivityVO.setFwlxString(sysDictData.getDictLabel());
            }
        }
        return success(partyActivityVO);
    }

    /**
     * 新增党员活动
     */
    @Log(title = "党员活动", businessType = BusinessType.INSERT)
    @PostMapping("/partyActivityInsertOrUpdate")
    public AjaxResult add(@RequestBody PartyActivity partyActivity)
    {
        return toAjax(partyActivityService.insertOrUpdatePartyActivity(partyActivity));
    }

    /**
     * 修改党员活动
     */
    @Log(title = "党员活动", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody PartyActivity partyActivity)
    {
        return toAjax(partyActivityService.updatePartyActivity(partyActivity));
    }

    /**
     * 删除党员活动
     */
    @Log(title = "党员活动", businessType = BusinessType.DELETE)
	@DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable String[] ids)
    {
        return toAjax(partyActivityService.deletePartyActivityByIds(ids));
    }

    //党组织活动
    @RequestMapping(value="getPartyActivity",method = RequestMethod.GET)
    @ResponseBody
    public AjaxResult getPartyActivity(String orgId){
        List<SysDictData> sysDictData = getDataByType();
        Map<String, SysDictData> treeMap = sysDictData.stream().collect(Collectors.toMap(SysDictData::getDictValue, x -> x));
        List<Map<String,Object>> list= partyActivityService.getPartyActivity(orgId);
        if(!CollectionUtils.isEmpty(list)){
            for(Map<String, Object> map : list){
                String fwlx = map.get("fwlx").toString();
                if(treeMap.containsKey(fwlx)){
                    map.put("fwlx", treeMap.get(fwlx).getDictLabel());
                }
            }
        }
        return success(list);
    }

    //党组织活动排行榜，没数据的依旧返回
    @RequestMapping(value="getPartyActivityRank",method = RequestMethod.GET)
    @ResponseBody
    public AjaxResult getPartyActivityRank(String orgId){
        List<SysDictData> sysDictData = getDataByType();
        Map<String, String> treeMap = sysDictData.stream().collect(Collectors.toMap(SysDictData::getDictValue, SysDictData::getDictLabel));
        List<String> keyList = Lists.newArrayList(treeMap.keySet());
        List<Map<String,Object>> list= partyActivityService.getPartyActivity(orgId);
        List<String> hasDataList = new ArrayList<>();
        if(!CollectionUtils.isEmpty(list)){
            for(Map<String, Object> map : list){
                String fwlx = map.get("fwlx").toString();
                if(treeMap.containsKey(fwlx)){
                    map.put("fwlx", treeMap.get(fwlx));
                    hasDataList.add(fwlx);
                }
            }
            keyList.removeAll(hasDataList);
        }
        if(!CollectionUtils.isEmpty(keyList)){
            for(String key : keyList){
                Map<String, Object> map = new HashMap<>();
                map.put("count",0);
                map.put("fwlx", treeMap.get(key));
                list.add(map);
            }
        }
        return success(list);
    }


}   
