
package com.ruoyi.web.controller.grid;

import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.system.domain.grid.GridCompanyEconomyInfo;
import com.ruoyi.system.domain.grid.ext.GridCompanyEconomyInfoExt;
import com.ruoyi.system.service.grid.GridCompanyEconomyInfoService;
import io.swagger.annotations.Api;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 企业经济信息
 */
@Api(tags = "数据看板 - 企业经济信息")
@RestController
@RequestMapping("/api/grid/company/economyIfo")
public class CompanyEconomyInfoController {

    @Autowired
    private GridCompanyEconomyInfoService companyEconomyInfoService;

    @PostMapping(value = "import", headers = "content-type=multipart/form-data")
    public AjaxResult importExcel(@RequestParam(value = "file") MultipartFile file) {
        return companyEconomyInfoService.importExcel(file);
    }

    /**
     * 不分页查询企业经济信息
     *
     * @param companyId
     * @return
     */
    @GetMapping(value = "/list")
    public AjaxResult list(@RequestParam(name = "companyId") String companyId) {

        Map<String, Object> result = new HashMap<>();
        List<GridCompanyEconomyInfo> infos = companyEconomyInfoService.listEconomyInfos(companyId, null);
        result.put("list", infos);
        return AjaxResult.success(result);
    }

    /**
     * 分页查询企业经济信息
     */
    @GetMapping(value = "/page")
    public AjaxResult page(@RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                           @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                           @RequestParam(value = "companyKeyword", required = false) String companyKeyword,
                           @RequestParam(value = "qylxCode", required = false) String qylxCode,
                           @RequestParam(value = "year", required = false) String year,
                           @RequestParam(value = "orderType", defaultValue = "desc") String orderType,
                           @RequestParam(value = "orderValue", defaultValue = "year") String orderValue) {
        return AjaxResult.success(companyEconomyInfoService.pageEconomyInfos(pageNo, pageSize, year, qylxCode, companyKeyword, orderValue, orderType));
    }

    /**
     * 检查是否需要覆盖
     */
    @GetMapping(value = "/check")
    public AjaxResult check(@RequestParam(value = "year") String year,
                            @RequestParam(name = "companyId") String companyId) {
        if (StringUtils.isBlank(year) || StringUtils.isBlank(companyId)) {
            return AjaxResult.error(400, "缺少必要参数");
        }
        return AjaxResult.success(companyEconomyInfoService.checkIsExist(year, companyId));
    }

    /**
     * 保存
     */
    @PostMapping(value = "/save")
    public AjaxResult save(@RequestBody GridCompanyEconomyInfoExt infoExt) {
        companyEconomyInfoService.save(infoExt);
        return AjaxResult.success("提交成功");
    }

    /**
     * 详情
     */
    @GetMapping(value = "/detail/{id}")
    public AjaxResult detail(@PathVariable("id") String id) {
        return AjaxResult.success(companyEconomyInfoService.detail(id));
    }

    /**
     * 删除
     */
    @GetMapping(value = "/delete/{id}")
    public AjaxResult delete(@PathVariable("id") String id) {
        companyEconomyInfoService.delete(id);
        return AjaxResult.success("删除成功");
    }

    /**
     * 亩均经济比例统计
     */
    @GetMapping(value = "/perMu/count")
    public AjaxResult perMuCount(@RequestParam(value = "year") String year) {
        if (StringUtils.isBlank(year)) {
            return AjaxResult.error(400, "缺少必要参数");
        }
        return AjaxResult.success(companyEconomyInfoService.perMuCount(year));
    }

    /**
     * 村级销售排名统计
     */
    @GetMapping(value = "/sale/count")
    public AjaxResult saleCount(@RequestParam(value = "year") String year) {
        if (StringUtils.isBlank(year)) {
            return AjaxResult.error(400, "缺少必要参数");
        }
        return AjaxResult.success(companyEconomyInfoService.saleCount(year));
    }

    /**
     * 村级税收排名统计
     */
    @GetMapping(value = "/tax/count")
    public AjaxResult taxCount(@RequestParam(value = "year") String year) {
        if (StringUtils.isBlank(year)) {
            return AjaxResult.error(400, "缺少必要参数");
        }
        return AjaxResult.success(companyEconomyInfoService.taxCount(year));
    }


    /**
     * 地块信息统计
     */
    @GetMapping(value = "/group/info")
    public AjaxResult groupInfo(@RequestParam(value = "landId") String landId, @RequestParam(value = "year") String year) {
        if (StringUtils.isBlank(year) || StringUtils.isBlank(landId)) {
            return AjaxResult.error(400, "缺少必要参数");
        }
        return AjaxResult.success(companyEconomyInfoService.loadCompanyLandInfo(landId, year));
    }
}