package com.ruoyi.common.core.redis;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.*;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * spring redis 工具类
 *
 * @author ruoyi
 **/
@SuppressWarnings(value = { "unchecked", "rawtypes" })
@Component
public class PublicServiceRedisCache
{
    @Autowired
    public RedisTemplate redisTemplate;

    /**
     * 缓存基本的对象，Integer、String、实体类等
     *
     * @param key 缓存的键值
     * @param value 缓存的值
     */
    public <T> void setCacheObject(final String key, final T value)
    {
        redisTemplate.opsForValue().set(key, value);
    }

    /**
     * 缓存基本的对象，Integer、String、实体类等
     *
     * @param key 缓存的键值
     * @param value 缓存的值
     * @param timeout 时间
     * @param timeUnit 时间颗粒度
     */
    public <T> void setCacheObject(final String key, final T value, final Integer timeout, final TimeUnit timeUnit)
    {
        redisTemplate.opsForValue().set(key, value, timeout, timeUnit);
    }

    /**
     * 设置有效时间
     *
     * @param key Redis键
     * @param timeout 超时时间
     * @return true=设置成功；false=设置失败
     */
    public boolean expire(final String key, final long timeout)
    {
        return expire(key, timeout, TimeUnit.SECONDS);
    }

    /**
     * 设置有效时间
     *
     * @param key Redis键
     * @param timeout 超时时间
     * @param unit 时间单位
     * @return true=设置成功；false=设置失败
     */
    public boolean expire(final String key, final long timeout, final TimeUnit unit)
    {
        return redisTemplate.expire(key, timeout, unit);
    }

    /**
     * 获得缓存的基本对象。
     *
     * @param key 缓存键值
     * @return 缓存键值对应的数据
     */
    public <T> T getCacheObject(final String key)
    {
        ValueOperations<String, T> operation = redisTemplate.opsForValue();
        return operation.get(key);
    }

    /**
     * 删除单个对象
     *
     * @param key
     */
    public boolean deleteObject(final String key)
    {
        return redisTemplate.delete(key);
    }
    public long deleteObject(@NotBlank final String key,@NotBlank final Object... hkey)
    {
        return redisTemplate.opsForHash().delete(key,hkey);
    }

    /**
     * 删除集合对象
     *
     * @param collection 多个对象
     * @return
     */
    public long deleteObject(final Collection collection)
    {
        return redisTemplate.delete(collection);
    }

    /**
     * 缓存List数据
     *
     * @param key 缓存的键值
     * @param dataList 待缓存的List数据
     * @return 缓存的对象
     */
    public <T> long setCacheList(final String key, final List<T> dataList)
    {
        Long count = redisTemplate.opsForList().rightPushAll(key, dataList);
        return count == null ? 0 : count;
    }

    /**
     * 获得缓存的list对象
     *
     * @param key 缓存的键值
     * @return 缓存键值对应的数据
     */
    public <T> List<T> getCacheList(final String key)
    {
        return redisTemplate.opsForList().range(key, 0, -1);
    }

    /**
     * 缓存Set
     *
     * @param key 缓存键值
     * @param dataSet 缓存的数据
     * @return 缓存数据的对象
     */
    public <T> long setCacheSet(final String key, final Set<T> dataSet)
    {
        Long count = redisTemplate.opsForSet().add(key, dataSet);
        return count == null ? 0 : count;
    }

    /**
     * 获得缓存的set
     *
     * @param key
     * @return
     */
    public <T> Set<T> getCacheSet(final String key)
    {
        return redisTemplate.opsForSet().members(key);
    }

    /**
     * 缓存Map
     *
     * @param key
     * @param dataMap
     */
    public <T> void setCacheMap(final String key, final Map<String, T> dataMap)
    {
        if (dataMap != null) {
            redisTemplate.opsForHash().putAll(key, dataMap);
        }
    }

    /**
     * 获得缓存的Map
     *
     * @param key
     * @return
     */
    public <T> Map<String, T> getCacheMap(final String key)
    {
        return redisTemplate.opsForHash().entries(key);
    }

    /**
     * 往Hash中存入数据
     *
     * @param key Redis键
     * @param hKey Hash键
     * @param value 值
     */
    public <T> void setCacheMapValue(final String key, final String hKey, final T value)
    {
        redisTemplate.opsForHash().put(key, hKey, value);
    }

    /**
     * 获取Hash中的数据
     *
     * @param key Redis键
     * @param hKey Hash键
     * @return Hash中的对象
     */
    public <T> T getCacheMapValue(final String key, final String hKey)
    {
        HashOperations<String, String, T> opsForHash = redisTemplate.opsForHash();
        return opsForHash.get(key, hKey);
    }

    /**
     * 获取多个Hash中的数据
     *
     * @param key Redis键
     * @param hKeys Hash键集合
     * @return Hash对象集合
     */
    public <T> List<T> getMultiCacheMapValue(final String key, final Collection<Object> hKeys)
    {
        return redisTemplate.opsForHash().multiGet(key, hKeys);
    }

    /**
     * 是否存在数据
     */
    public boolean exists(@NotBlank final String key) {
        return redisTemplate.hasKey(key);
    }
    public boolean exists(@NotBlank final String key,@NotBlank final String hkey) {
        return redisTemplate.opsForHash().hasKey(key,hkey);
    }

    /**
     * 事务命令
     * @param sessionCallback
     * @return
     */
    public Object muiti(SessionCallback sessionCallback) {
        return redisTemplate.execute(sessionCallback);
    }



    /**
     * 获取缓存的时间
     */
    public long getCacheTTL(final String key) {
        return redisTemplate.getExpire(key);
    }

    private static RedisCache instance;
    public PublicServiceRedisCache() {
    }
    public static RedisCache getInstance() {
        return new RedisCache();
    }
    private final Keys keys = new Keys();
    private final Strings strings = new Strings();
    private final Lists lists = new Lists();
    private final Sets sets = new Sets();
    private final Hashs hashs = new Hashs();


    public Keys keys() {
        return keys;
    }

    public Strings strings() {
        return strings;
    }

    public Lists lists() {
        return lists;
    }

    public Sets sets() {
        return sets;
    }

    public Hashs hashs() {
        return hashs;
    }


    public class Keys {
        /**
         * 设置过期时间
         * @param key 键值
         * @param timeout 时长
         * @param timeUnit 时间单位
         * @return
         */
        public boolean expire(@NotBlank String key, long timeout, TimeUnit timeUnit) {
            return redisTemplate.expire(key,timeout,timeUnit);
        }

        /**
         * 查询有效期
         * @param key 键值
         * @return
         */
        public long ttl(@NotBlank String key) {
            return redisTemplate.getExpire(key);
        }
        /**
         * 取消有效期设置
         * @param key 键值
         * @return
         */
        public boolean expire(@NotBlank String key) {
            return redisTemplate.persist(key);
        }

        /**
         * 判断key是否存在
         * @param key 键值
         * @return
         */
        public boolean exists(@NotBlank String key) {
            return redisTemplate.hasKey(key);
        }

        /**
         * 删除key，可批量
         * @param keys 键值集合
         * @return
         */
        public long del(Collection keys) {
            return redisTemplate.delete(keys);
        }
        public boolean del(Object keys) {
            return redisTemplate.delete(keys);
        }

        /**
         * 查询key的类型
         * @param key
         * @return
         */
        public String type(@NotBlank String key) {
            return redisTemplate.type(key).name();
        }

        /**
         * 查询键值集合，消耗性能，非必要不使用
         * @param pattern 键值匹配，如：key:*
         * @return
         */
        public Collection<String> keys(@NotBlank String pattern) {
            return redisTemplate.keys(pattern);
        }

    }

    public class Sets {

        /**
         * 添加元素
         * @param key
         * @param val
         * @return
         */
        public <K,V> Long add(K key,V... val) {
            SetOperations<K,V> s = redisTemplate.opsForSet();
           return s.add(key,val);
        }


    }

    public class Hashs {

        /**
         *
         * @param key
         * @param hk
         * @param hv
         * @param <T>
         */
        public <T> void put(@NotBlank final String key,@NotBlank final Object hk,Object hv) {
            redisTemplate.opsForHash().put(key,hk,hv);
        }

        public <T> void put(@NotBlank final String key,Map map) {
            redisTemplate.opsForHash().putAll(key,map);
        }

        /**
         * 获取hash中指定的key的值
         * @param key
         * @param hk
         * @param <Hk>
         * @param <HV>
         * @return
         */
        public <Hk, HV> HV get(@NotBlank final String key,@NotBlank final Object hk) {
            HashOperations<String, Hk, HV> operation = redisTemplate.opsForHash();
            return operation.get(key,hk);
        }

        /**
         * 获取map集合
         * @param key
         * @param <Hk>
         * @param <HV>
         * @return
         */
        public <Hk, HV> Map get(@NotBlank final String key) {
            HashOperations<String, Hk, HV> operation = redisTemplate.opsForHash();
            return operation.entries(key);
        }


    }

    public class Strings {
        /**
         * 根据key获取记录
         *
         * @param key
         * @return 值
         */
        public <T> T get(@NotBlank final String key)
        {
            ValueOperations<String, T> operation = redisTemplate.opsForValue();
            return operation.get(key);
        }

        /**
         * 设置键的值
         * @param key
         * @param value
         */
        public  <T> void set(@NotBlank String key, @NotBlank T value) {
            redisTemplate.opsForValue().set(key,value);
        }

        /**
         * 设置键的值并指定失效时间
         * @param key
         * @param value
         * @param timeout 过期时间
         * @param timeUnit 时间单位
         */
        public <T> void set(@NotBlank String key, @NotBlank T value,final long timeout,final TimeUnit timeUnit) {
            redisTemplate.opsForValue().set(key,value,timeout,timeUnit);
        }
        /**
         * 设置键的值 如果键不存在则新增,存在则不改变已经有的值
         * @param key
         * @param value
         */
        public <T> boolean setnx(@NotBlank String key,@NotBlank T value) {
            return redisTemplate.opsForValue().setIfAbsent(key,value);
        }
        public <T> boolean setnx(@NotBlank String key,@NotBlank T value,final long timeout,final TimeUnit timeUnit) {
            return redisTemplate.opsForValue().setIfAbsent(key,value,timeout,timeUnit);
        }

        /**
         * 设置map集合到redis
         * @param map
         */
        public <K,V> void multiSet(@NotBlank Map<? extends K, ? extends V> map) {
            ValueOperations<K,V> operation = redisTemplate.opsForValue();
            operation.multiSet(map);
        }

        public boolean multiSetIfExists(@NotBlank Map map) {
            return redisTemplate.opsForValue().multiSetIfAbsent(map);
        }
        /**
         * 增加1
         * @param key
         * @return
         */
        public long incr(@NotBlank String key) {
            return redisTemplate.opsForValue().increment(key);
        }

        /**
         * key值增加指定的值，值为负数时未减去对应的值
         * @param key
         * @param value
         * @return
         */
        public long incrBy(@NotBlank String key,final long value) {
            return redisTemplate.opsForValue().increment(key,value);
        }
        public double incrBy(@NotBlank String key,final double value) {
            return redisTemplate.opsForValue().increment(key,value);
        }
        /**
         * 增加1
         * @param key
         * @return
         */
        public long decr(@NotBlank String key) {
            return this.incrBy(key,-1);
        }

        /**
         * key值增加指定的值，值为负数时未减去对应的值
         * @param key
         * @param value
         * @return
         */
        public long decrBy(@NotBlank String key,final long value) {
            return this.incrBy(key,-value);
        }
    }

    public class Lists {

    }

}
