package com.ruoyi.web.controller.system;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletResponse;

import cn.hutool.crypto.KeyUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.symmetric.DES;
import cn.hutool.crypto.symmetric.SymmetricAlgorithm;
import com.alibaba.fastjson.JSON;
import com.ruoyi.common.core.domain.DeptUserCountTree;
import com.ruoyi.common.core.domain.DeptUserTree;
import com.ruoyi.common.core.domain.TreeSelect;
import com.ruoyi.common.utils.bean.BeanCopyUtil;
import com.ruoyi.common.utils.uuid.IdUtils;
import com.ruoyi.generator.utils.AesUtil;
import io.swagger.annotations.Api;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.domain.entity.SysDept;
import com.ruoyi.common.core.domain.entity.SysRole;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.SecurityUtils;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.system.service.system.ISysDeptService;
import com.ruoyi.system.service.system.ISysPostService;
import com.ruoyi.system.service.system.ISysRoleService;
import com.ruoyi.system.service.system.ISysUserService;

/**
 * 用户信息
 * 
 * @author ruoyi
 */
@Api(tags = "基础服务 - 用户信息")
@RestController
@RequestMapping("/system/user")
public class SysUserController extends BaseController
{
    @Autowired
    private ISysUserService userService;

    @Autowired
    private ISysRoleService roleService;

    @Autowired
    private ISysDeptService deptService;

    @Autowired
    private ISysPostService postService;

    private static final String SECRET = "Av627+rvCwc=";
    /**
     * 获取用户列表
     */
    @GetMapping("/list")
    public TableDataInfo list(SysUser user)
    {
        startPage();
        List<SysUser> list = userService.selectUserList(user);
        return getDataTable(list);
    }

    /**
     * APP专用人员列表
     * @return
     */
    @GetMapping("/listForApp")
    public AjaxResult listForApp(@RequestParam(value = "deptId", required = false)Long deptId,
                                 @RequestParam(value = "nickName", required = false)String nickName)
    {
        SysUser sysUser = new SysUser();
        sysUser.setNickName(nickName);
        sysUser.setStatus("0");
        sysUser.setDeptId(deptId);
        return AjaxResult.success(userService.selectUserList(sysUser));
    }

    /**
     * 人员列表 此接口忽略token但数据需要加密
     * @return
     */
    @GetMapping("/listWithoutToken")
    public AjaxResult listWithoutToken()
    {
        List<SysUser> sysUsers = userService.listAllUsers();
        //精简字段
        List<String> result = new ArrayList<>();
        sysUsers.forEach(user -> {
            SysUser sysUser = new SysUser();
            sysUser.setUserId(user.getUserId());
            sysUser.setNickName(user.getNickName());
            sysUser.setUserName(user.getUserName());
            try {
                DES des = SecureUtil.des(AesUtil.base64Decode(SECRET));
                result.add(des.encryptHex(JSON.toJSONString(sysUser)));
            } catch (Exception e) {
                e.printStackTrace();
            }
        });
        return AjaxResult.success(result);
    }


    @Log(title = "用户管理", businessType = BusinessType.EXPORT)
    @PreAuthorize("@ss.hasPermi('system:user:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysUser user)
    {
        List<SysUser> list = userService.selectUserList(user);
        ExcelUtil<SysUser> util = new ExcelUtil<SysUser>(SysUser.class);
        util.exportExcel(response, list, "用户数据");
    }

    @Log(title = "用户管理", businessType = BusinessType.IMPORT)
    @PreAuthorize("@ss.hasPermi('system:user:import')")
    @PostMapping("/importData")
    public AjaxResult importData(MultipartFile file, boolean updateSupport) throws Exception
    {
        ExcelUtil<SysUser> util = new ExcelUtil<SysUser>(SysUser.class);
        List<SysUser> userList = util.importExcel(file.getInputStream());
        String operName = getUsername();
        String message = userService.importUser(userList, updateSupport, operName);
        return success(message);
    }

    @PostMapping("/importTemplate")
    public void importTemplate(HttpServletResponse response)
    {
        ExcelUtil<SysUser> util = new ExcelUtil<SysUser>(SysUser.class);
        util.importTemplateExcel(response, "用户数据");
    }

    /**
     * 根据用户编号获取详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:user:query')")
    @GetMapping(value = { "/", "/{userId}" })
    public AjaxResult getInfo(@PathVariable(value = "userId", required = false) String userId)
    {
        userService.checkUserDataScope(userId);
        AjaxResult ajax = AjaxResult.success();
        List<SysRole> roles = roleService.selectRoleAll();
        ajax.put("roles", SysUser.isAdmin(userId) ? roles : roles.stream().filter(r -> !r.isAdmin()).collect(Collectors.toList()));
        ajax.put("posts", postService.selectPostAll());
        if (StringUtils.isNotNull(userId))
        {
            SysUser sysUser = userService.selectUserById(userId);
            ajax.put(AjaxResult.DATA_TAG, sysUser);
            ajax.put("postIds", postService.selectPostListByUserId(userId));
            ajax.put("roleIds", sysUser.getRoles().stream().map(SysRole::getRoleId).collect(Collectors.toList()));
        }
        return ajax;
    }

    /**
     * 新增用户
     */
    @PreAuthorize("@ss.hasPermi('system:user:add')")
    @Log(title = "用户管理", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@Validated @RequestBody SysUser user)
    {
        if (!userService.checkUserNameUnique(user))
        {
            return error("新增用户'" + user.getUserName() + "'失败，登录账号已存在");
        }
        else if (StringUtils.isNotEmpty(user.getPhonenumber()) && !userService.checkPhoneUnique(user))
        {
            return error("新增用户'" + user.getUserName() + "'失败，手机号码已存在");
        }
        else if (StringUtils.isNotEmpty(user.getEmail()) && !userService.checkEmailUnique(user))
        {
            return error("新增用户'" + user.getUserName() + "'失败，邮箱账号已存在");
        }
        user.setUserId(IdUtils.simpleUUID());
        user.setCreateBy(getUsername());
        user.setPassword(SecurityUtils.encryptPassword(user.getPassword()));
        return toAjax(userService.insertUser(user));
    }

    /**
     * 修改用户
     */
    @PreAuthorize("@ss.hasPermi('system:user:edit')")
    @Log(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@Validated @RequestBody SysUser user)
    {
        userService.checkUserAllowed(user);
        userService.checkUserDataScope(user.getUserId());
        if (!userService.checkUserNameUnique(user))
        {
            return error("修改用户'" + user.getUserName() + "'失败，登录账号已存在");
        }
        else if (StringUtils.isNotEmpty(user.getPhonenumber()) && !userService.checkPhoneUnique(user))
        {
            return error("修改用户'" + user.getUserName() + "'失败，手机号码已存在");
        }
        else if (StringUtils.isNotEmpty(user.getEmail()) && !userService.checkEmailUnique(user))
        {
            return error("修改用户'" + user.getUserName() + "'失败，邮箱账号已存在");
        }
        user.setUpdateBy(getUsername());
        return toAjax(userService.updateUser(user));
    }

    /**
     * 删除用户
     */
    @PreAuthorize("@ss.hasPermi('system:user:remove')")
    @Log(title = "用户管理", businessType = BusinessType.DELETE)
    @DeleteMapping("/{userIds}")
    public AjaxResult remove(@PathVariable String[] userIds)
    {
        if (ArrayUtils.contains(userIds, getUserId()))
        {
            return error("当前用户不能删除");
        }
        return toAjax(userService.deleteUserByIds(userIds));
    }

    /**
     * 重置密码
     */
    @PreAuthorize("@ss.hasPermi('system:user:resetPwd')")
    @Log(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping("/resetPwd")
    public AjaxResult resetPwd(@RequestBody SysUser user)
    {
        userService.checkUserAllowed(user);
        userService.checkUserDataScope(user.getUserId());
        user.setPassword(SecurityUtils.encryptPassword(user.getPassword()));
        user.setUpdateBy(getUsername());
        return toAjax(userService.resetPwd(user));
    }

    /**
     * 状态修改
     */
    @PreAuthorize("@ss.hasPermi('system:user:edit')")
    @Log(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping("/changeStatus")
    public AjaxResult changeStatus(@RequestBody SysUser user)
    {
        userService.checkUserAllowed(user);
        userService.checkUserDataScope(user.getUserId());
        user.setUpdateBy(getUsername());
        return toAjax(userService.updateUserStatus(user));
    }

    /**
     * 根据用户编号获取授权角色
     */
    @PreAuthorize("@ss.hasPermi('system:user:query')")
    @GetMapping("/authRole/{userId}")
    public AjaxResult authRole(@PathVariable("userId") String userId)
    {
        AjaxResult ajax = AjaxResult.success();
        SysUser user = userService.selectUserById(userId);
        List<SysRole> roles = roleService.selectRolesByUserId(userId);
        ajax.put("user", user);
        ajax.put("roles", SysUser.isAdmin(userId) ? roles : roles.stream().filter(r -> !r.isAdmin()).collect(Collectors.toList()));
        return ajax;
    }

    /**
     * 用户授权角色
     */
    @PreAuthorize("@ss.hasPermi('system:user:edit')")
    @Log(title = "用户管理", businessType = BusinessType.GRANT)
    @PutMapping("/authRole")
    public AjaxResult insertAuthRole(String userId, Long[] roleIds)
    {
        userService.checkUserDataScope(userId);
        userService.insertUserAuth(userId, roleIds);
        return success();
    }

    /**
     * 获取部门树列表
     */
    @GetMapping("/deptTree")
    public AjaxResult deptTree(SysDept dept)
    {
        return success(deptService.selectDeptTreeList(dept));
    }

    /**
     * APP部门树（含人员数量）
     * @return
     */
    @GetMapping("/deptTreeWithUserCount")
    public AjaxResult deptTreeWithCount()
    {
        SysDept dept = new SysDept();
        dept.setNeedAuth(false);
        dept.setStatus("0");
        List<TreeSelect> treeSelectList = deptService.selectDeptTreeList(dept);
        List<DeptUserCountTree> userTrees = BeanCopyUtil.bathToBean(treeSelectList, DeptUserCountTree.class);
        if(!CollectionUtils.isEmpty(userTrees)){
            setDepUserWithCountChild(userTrees);
        }
        return AjaxResult.success(userTrees);
    }

    private void setDepUserWithCountChild(List<DeptUserCountTree> userTrees){
        for(DeptUserCountTree userTree : userTrees){
            int count = deptService.countUserByDeptId(Long.parseLong(userTree.getId()));
            userTree.setUserCount(count);
            if(!CollectionUtils.isEmpty(userTree.getChildren())){
                setDepUserWithCountChild(userTree.getChildren());
            }
        }
    }

    /**
     * 包含部门的用户树
     * @return
     */
    @GetMapping("/userTree")
    public AjaxResult getUserTree(){
        SysDept dept = new SysDept();
        dept.setStatus("0");
        dept.setNeedAuth(false);
        List<TreeSelect> deptTree = deptService.selectDeptTreeList(dept);
        List<DeptUserTree> userTrees = BeanCopyUtil.bathToBean(deptTree, DeptUserTree.class);
        if(!CollectionUtils.isEmpty(userTrees)){
            setDeptUserChild(userTrees);
        }
        return AjaxResult.success(userTrees);
    }

    private void setDeptUserChild(List<DeptUserTree> userTrees){
        for(DeptUserTree userTree : userTrees){
            int count = deptService.countUserByDeptId(Long.parseLong(userTree.getId()));
            userTree.setCount(count);
            List<SysUser> sysUsers = userService.listByDepIdOnly(Long.parseLong(userTree.getId()));
            List<DeptUserTree> treeList = new ArrayList<>();
            if(!CollectionUtils.isEmpty(sysUsers)){
                for(SysUser sysUser : sysUsers){
                    DeptUserTree tree = new DeptUserTree();
                    tree.setIsDep(false);
                    tree.setAvatar(sysUser.getAvatar());
                    tree.setId(sysUser.getUserId());
                    tree.setLabel(sysUser.getNickName());
                    treeList.add(tree);
                }
            }
            if(!CollectionUtils.isEmpty(userTree.getChildren())){
                setDeptUserChild(userTree.getChildren());
                userTree.getChildren().addAll(treeList);
            }else{
                userTree.setChildren(treeList);
            }
        }
    }


}
