/**
 * Node.js Hello World 应用
 * @description 一个简单的 HTTP 服务器，返回 Hello World 消息
 */

const http = require('http');

/**
 * 创建 HTTP 服务器
 * @param {http.IncomingMessage} req - HTTP 请求对象
 * @param {http.ServerResponse} res - HTTP 响应对象
 */
const requestHandler = (req, res) => {
  // 设置响应头
  res.setHeader('Content-Type', 'text/html; charset=utf-8');
  res.setHeader('Access-Control-Allow-Origin', '*');
  
  // 根据请求路径返回不同内容
  if (req.url === '/') {
    res.writeHead(200);
    res.end(`
      <!DOCTYPE html>
      <html lang="zh-CN">
      <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Node.js Hello World</title>
        <style>
          body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            min-height: 100vh;
          }
          .container {
            background: rgba(255, 255, 255, 0.1);
            padding: 30px;
            border-radius: 15px;
            backdrop-filter: blur(10px);
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
          }
          h1 {
            text-align: center;
            margin-bottom: 30px;
            font-size: 2.5em;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.3);
          }
          .info {
            background: rgba(255, 255, 255, 0.2);
            padding: 20px;
            border-radius: 10px;
            margin: 20px 0;
          }
          .endpoint {
            background: rgba(0, 0, 0, 0.2);
            padding: 15px;
            border-radius: 8px;
            font-family: monospace;
            margin: 10px 0;
          }
          .status {
            text-align: center;
            font-size: 1.2em;
            margin-top: 30px;
            padding: 15px;
            background: rgba(76, 175, 80, 0.3);
            border-radius: 8px;
          }
        </style>
      </head>
      <body>
        <div class="container">
          <h1>🚀 Node.js Hello World</h1>
          
          <div class="info">
            <h2>项目信息</h2>
            <p><strong>Node.js 版本:</strong> ${process.version}</p>
            <p><strong>平台:</strong> ${process.platform}</p>
            <p><strong>架构:</strong> ${process.arch}</p>
            <p><strong>启动时间:</strong> ${new Date().toLocaleString('zh-CN')}</p>
          </div>
          
          <div class="info">
            <h2>可用端点</h2>
            <div class="endpoint">GET / - 显示此页面</div>
            <div class="endpoint">GET /api/hello - 返回 JSON 格式的 Hello World</div>
            <div class="endpoint">GET /api/status - 返回服务器状态信息</div>
          </div>
          
          <div class="status">
            ✅ 服务器运行正常！当前时间: ${new Date().toLocaleTimeString('zh-CN')}
          </div>
        </div>
      </body>
      </html>
    `);
  } else if (req.url === '/api/hello') {
    res.writeHead(200, { 'Content-Type': 'application/json' });
    res.end(JSON.stringify({
      message: 'Hello World!',
      timestamp: new Date().toISOString(),
      language: 'zh-CN'
    }));
  } else if (req.url === '/api/status') {
    res.writeHead(200, { 'Content-Type': 'application/json' });
    res.end(JSON.stringify({
      status: 'running',
      uptime: process.uptime(),
      memory: process.memoryUsage(),
      platform: process.platform,
      nodeVersion: process.version
    }));
  } else {
    res.writeHead(404);
    res.end(JSON.stringify({
      error: 'Not Found',
      message: '请求的路径不存在',
      availableEndpoints: ['/', '/api/hello', '/api/status']
    }));
  }
};

/**
 * 创建 HTTP 服务器实例
 */
const server = http.createServer(requestHandler);

/**
 * 服务器端口配置
 */
const PORT = process.env.PORT || 3000;

/**
 * 启动服务器
 */
server.listen(PORT, () => {
  console.log(`🚀 服务器启动成功！`);
  console.log(`📍 访问地址: http://localhost:${PORT}`);
  console.log(`⏰ 启动时间: ${new Date().toLocaleString('zh-CN')}`);
  console.log(`🔧 Node.js 版本: ${process.version}`);
  console.log(`💻 平台: ${process.platform} ${process.arch}`);
});

/**
 * 优雅关闭服务器
 */
process.on('SIGTERM', () => {
  console.log('收到 SIGTERM 信号，正在关闭服务器...');
  server.close(() => {
    console.log('服务器已关闭');
    process.exit(0);
  });
});

process.on('SIGINT', () => {
  console.log('收到 SIGINT 信号，正在关闭服务器...');
  server.close(() => {
    console.log('服务器已关闭');
    process.exit(0);
  });
});
