/**
 * Three.js 项目模板
 * 包含基础 3D 场景、光照、材质和交互功能
 */

import * as THREE from 'three';

/**
 * 主应用类
 * 负责管理 Three.js 场景、渲染器和交互控制
 */
class ThreeJSApp {
    constructor() {
        this.scene = null;
        this.camera = null;
        this.renderer = null;
        this.controls = null;
        this.meshes = [];
        this.clock = new THREE.Clock();
        this.frameCount = 0;
        this.lastTime = 0;
        this.stats = {
            fps: 0,
            objects: 0,
            vertices: 0
        };
        
        this.init();
    }

    /**
     * 初始化 Three.js 应用
     */
    init() {
        this.createScene();
        this.createCamera();
        this.createRenderer();
        this.createLights();
        this.createGeometry();
        this.setupControls();
        this.setupEventListeners();
        this.hideLoading();
        this.updateStats(); // 初始化统计信息
        this.animate();
    }

    /**
     * 创建场景
     */
    createScene() {
        this.scene = new THREE.Scene();
        this.scene.background = new THREE.Color(0x222222);
        this.scene.fog = new THREE.Fog(0x222222, 10, 100);
    }

    /**
     * 创建相机
     */
    createCamera() {
        this.camera = new THREE.PerspectiveCamera(
            75,
            window.innerWidth / window.innerHeight,
            0.1,
            1000
        );
        this.camera.position.set(5, 5, 5);
        this.camera.lookAt(0, 0, 0);
    }

    /**
     * 创建渲染器
     */
    createRenderer() {
        this.renderer = new THREE.WebGLRenderer({ 
            antialias: true,
            alpha: true 
        });
        this.renderer.setSize(window.innerWidth, window.innerHeight);
        this.renderer.setPixelRatio(window.devicePixelRatio);
        this.renderer.shadowMap.enabled = true;
        this.renderer.shadowMap.type = THREE.PCFSoftShadowMap;
        this.renderer.outputColorSpace = THREE.SRGBColorSpace;
        
        document.getElementById('container').appendChild(this.renderer.domElement);
    }

    /**
     * 创建光照系统
     */
    createLights() {
        // 环境光
        const ambientLight = new THREE.AmbientLight(0x404040, 0.6);
        this.scene.add(ambientLight);

        // 方向光（太阳光）
        const directionalLight = new THREE.DirectionalLight(0xffffff, 1);
        directionalLight.position.set(10, 10, 5);
        directionalLight.castShadow = true;
        directionalLight.shadow.mapSize.width = 2048;
        directionalLight.shadow.mapSize.height = 2048;
        directionalLight.shadow.camera.near = 0.5;
        directionalLight.shadow.camera.far = 50;
        directionalLight.shadow.camera.left = -10;
        directionalLight.shadow.camera.right = 10;
        directionalLight.shadow.camera.top = 10;
        directionalLight.shadow.camera.bottom = -10;
        this.scene.add(directionalLight);

        // 点光源
        const pointLight = new THREE.PointLight(0x3498db, 0.8, 100);
        pointLight.position.set(-5, 5, 5);
        this.scene.add(pointLight);

        // 聚光灯
        const spotLight = new THREE.SpotLight(0xff6b6b, 0.5, 100, Math.PI / 6, 0.3);
        spotLight.position.set(0, 10, 0);
        spotLight.target.position.set(0, 0, 0);
        this.scene.add(spotLight);
        this.scene.add(spotLight.target);
    }

    /**
     * 创建几何体和材质
     */
    createGeometry() {
        // 创建各种几何体
        const geometries = [
            { type: 'box', position: [0, 1, 0], color: 0xff6b6b },
            { type: 'sphere', position: [-3, 1, 0], color: 0x4ecdc4 },
            { type: 'cone', position: [3, 1, 0], color: 0x45b7d1 },
            { type: 'cylinder', position: [0, 1, -3], color: 0x96ceb4 },
            { type: 'torus', position: [0, 1, 3], color: 0xfeca57 },
            { type: 'octahedron', position: [-3, 1, 3], color: 0xff9ff3 },
            { type: 'tetrahedron', position: [3, 1, 3], color: 0xffa726 },
            { type: 'dodecahedron', position: [-3, 1, -3], color: 0xab47bc },
            { type: 'icosahedron', position: [3, 1, -3], color: 0x26a69a }
        ];

        geometries.forEach((geo, index) => {
            let geometry;
            
            switch (geo.type) {
                case 'box':
                    geometry = new THREE.BoxGeometry(1, 1, 1);
                    break;
                case 'sphere':
                    geometry = new THREE.SphereGeometry(0.7, 32, 32);
                    break;
                case 'cone':
                    geometry = new THREE.ConeGeometry(0.7, 1.5, 32);
                    break;
                case 'cylinder':
                    geometry = new THREE.CylinderGeometry(0.7, 0.7, 1.5, 32);
                    break;
                case 'torus':
                    geometry = new THREE.TorusGeometry(0.7, 0.3, 16, 100);
                    break;
                case 'octahedron':
                    geometry = new THREE.OctahedronGeometry(0.7);
                    break;
                case 'tetrahedron':
                    geometry = new THREE.TetrahedronGeometry(0.7);
                    break;
                case 'dodecahedron':
                    geometry = new THREE.DodecahedronGeometry(0.7);
                    break;
                case 'icosahedron':
                    geometry = new THREE.IcosahedronGeometry(0.7);
                    break;
            }

            // 创建材质
            const material = new THREE.MeshPhongMaterial({
                color: geo.color,
                shininess: 100,
                transparent: true,
                opacity: 0.9
            });

            // 创建网格
            const mesh = new THREE.Mesh(geometry, material);
            mesh.position.set(...geo.position);
            mesh.castShadow = true;
            mesh.receiveShadow = true;
            mesh.userData = { 
                type: geo.type, 
                color: geo.color,
                index: index 
            };

            this.meshes.push(mesh);
            this.scene.add(mesh);
        });

        // 创建地面
        const groundGeometry = new THREE.PlaneGeometry(20, 20);
        const groundMaterial = new THREE.MeshLambertMaterial({ 
            color: 0x333333,
            transparent: true,
            opacity: 0.8
        });
        const ground = new THREE.Mesh(groundGeometry, groundMaterial);
        ground.rotation.x = -Math.PI / 2;
        ground.receiveShadow = true;
        this.scene.add(ground);

        // 更新统计信息
        this.updateStats();
    }

    /**
     * 设置控制器
     */
    setupControls() {
        // 简单的鼠标控制实现
        this.mouse = new THREE.Vector2();
        this.raycaster = new THREE.Raycaster();
        this.isMouseDown = false;
        this.mouseDownPosition = new THREE.Vector2();
        this.cameraPosition = new THREE.Vector3();
        
        // 保存初始相机位置
        this.cameraPosition.copy(this.camera.position);
    }

    /**
     * 设置事件监听器
     */
    setupEventListeners() {
        // 窗口大小调整
        window.addEventListener('resize', () => this.onWindowResize());
        
        // 鼠标事件
        this.renderer.domElement.addEventListener('mousedown', (event) => this.onMouseDown(event));
        this.renderer.domElement.addEventListener('mousemove', (event) => this.onMouseMove(event));
        this.renderer.domElement.addEventListener('mouseup', () => this.onMouseUp());
        this.renderer.domElement.addEventListener('wheel', (event) => this.onMouseWheel(event));
        this.renderer.domElement.addEventListener('click', (event) => this.onMouseClick(event));
    }

    /**
     * 鼠标按下事件
     */
    onMouseDown(event) {
        this.isMouseDown = true;
        this.mouseDownPosition.set(event.clientX, event.clientY);
    }

    /**
     * 鼠标移动事件
     */
    onMouseMove(event) {
        this.mouse.x = (event.clientX / window.innerWidth) * 2 - 1;
        this.mouse.y = -(event.clientY / window.innerHeight) * 2 + 1;

        if (this.isMouseDown) {
            const deltaX = event.clientX - this.mouseDownPosition.x;
            const deltaY = event.clientY - this.mouseDownPosition.y;
            
            // 旋转相机
            const spherical = new THREE.Spherical();
            spherical.setFromVector3(this.camera.position);
            spherical.theta -= deltaX * 0.01;
            spherical.phi += deltaY * 0.01;
            spherical.phi = Math.max(0.1, Math.min(Math.PI - 0.1, spherical.phi));
            
            this.camera.position.setFromSpherical(spherical);
            this.camera.lookAt(0, 0, 0);
            
            this.mouseDownPosition.set(event.clientX, event.clientY);
        }
    }

    /**
     * 鼠标释放事件
     */
    onMouseUp() {
        this.isMouseDown = false;
    }

    /**
     * 鼠标滚轮事件
     */
    onMouseWheel(event) {
        const scale = event.deltaY > 0 ? 1.1 : 0.9;
        this.camera.position.multiplyScalar(scale);
    }

    /**
     * 鼠标点击事件
     */
    onMouseClick(event) {
        this.raycaster.setFromCamera(this.mouse, this.camera);
        const intersects = this.raycaster.intersectObjects(this.meshes);
        
        if (intersects.length > 0) {
            const object = intersects[0].object;
            console.log('点击了物体:', object.userData);
            
            // 简单的点击反馈
            object.material.emissive.setHex(0x444444);
            setTimeout(() => {
                object.material.emissive.setHex(0x000000);
            }, 200);
        }
    }

    /**
     * 窗口大小调整
     */
    onWindowResize() {
        this.camera.aspect = window.innerWidth / window.innerHeight;
        this.camera.updateProjectionMatrix();
        this.renderer.setSize(window.innerWidth, window.innerHeight);
    }

    /**
     * 隐藏加载界面
     */
    hideLoading() {
        const loading = document.getElementById('loading');
        if (loading) {
            loading.style.display = 'none';
        }
    }

    /**
     * 更新统计信息
     */
    updateStats() {
        // 只统计网格对象，不包括光照、相机等
        let meshCount = 0;
        let vertices = 0;
        
        this.scene.traverse((child) => {
            if (child.isMesh && child.geometry) {
                meshCount++;
                if (child.geometry.attributes.position) {
                    vertices += child.geometry.attributes.position.count;
                }
            }
        });
        
        this.stats.objects = meshCount;
        this.stats.vertices = vertices;
        
        // 更新 DOM 显示
        document.getElementById('objects').textContent = this.stats.objects;
        document.getElementById('vertices').textContent = this.stats.vertices;
    }

    /**
     * 动画循环
     */
    animate() {
        requestAnimationFrame(() => this.animate());
        
        const elapsedTime = this.clock.getElapsedTime();
        const currentTime = performance.now();
        
        // 旋转所有网格
        this.meshes.forEach((mesh, index) => {
            mesh.rotation.x = elapsedTime * 0.5 + index * 0.5;
            mesh.rotation.y = elapsedTime * 0.3 + index * 0.3;
        });
        
        // 计算 FPS（使用更稳定的方法）
        this.frameCount++;
        if (currentTime >= this.lastTime + 1000) {
            this.stats.fps = Math.round((this.frameCount * 1000) / (currentTime - this.lastTime));
            document.getElementById('fps').textContent = this.stats.fps;
            this.frameCount = 0;
            this.lastTime = currentTime;
        }
        
        // 每60帧更新一次统计信息（减少计算频率）
        if (this.frameCount % 60 === 0) {
            this.updateStats();
        }
        
        // 渲染场景
        this.renderer.render(this.scene, this.camera);
    }
}

// 启动应用
document.addEventListener('DOMContentLoaded', () => {
    new ThreeJSApp();
});
