#!/bin/bash

# Three.js 项目启动脚本
# 用于 Docker 容器启动时的环境配置和服务启动

set -e

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 日志函数
log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 检查 Node.js 版本
check_node_version() {
    log_info "检查 Node.js 版本..."
    node_version=$(node --version)
    log_success "Node.js 版本: $node_version"
}

# 检查 npm 版本
check_npm_version() {
    log_info "检查 npm 版本..."
    npm_version=$(npm --version)
    log_success "npm 版本: $npm_version"
}

# 检查项目依赖
check_dependencies() {
    log_info "检查项目依赖..."
    if [ ! -f "package.json" ]; then
        log_error "package.json 文件不存在！"
        exit 1
    fi
    
    if [ ! -d "node_modules" ]; then
        log_warning "node_modules 目录不存在，正在安装依赖..."
        npm install
    fi
    
    log_success "依赖检查完成"
}

# 检查必要文件
check_required_files() {
    log_info "检查必要文件..."
    
    required_files=("index.html" "main.js" "styles.css")
    
    for file in "${required_files[@]}"; do
        if [ ! -f "$file" ]; then
            log_error "必要文件 $file 不存在！"
            exit 1
        fi
    done
    
    log_success "所有必要文件检查完成"
}

# 设置权限
set_permissions() {
    log_info "设置文件权限..."
    chmod -R 755 /app
    log_success "权限设置完成"
}

# 创建日志目录
create_log_dir() {
    log_info "创建日志目录..."
    mkdir -p /app/logs
    chmod 755 /app/logs
    log_success "日志目录创建完成"
}

# 启动健康检查
start_health_check() {
    log_info "启动健康检查..."
    
        # 在后台启动健康检查
    (
        sleep 5
        while true; do
            if curl -f http://localhost:${APP_PORT_1:-8000} > /dev/null 2>&1; then
                log_success "应用运行正常"
            else
                log_warning "应用可能未正常启动"
            fi
            sleep 30
        done
    ) &
    
    log_success "健康检查已启动"
}

# 显示启动信息
show_startup_info() {
    echo ""
    echo "=========================================="
    echo "    Three.js 项目模板启动成功！"
    echo "=========================================="
    echo ""
    echo "🌐 访问地址: http://localhost:${APP_PORT_1:-8000}"
    echo "📁 项目目录: /app"
    echo "📊 Node.js: $(node --version)"
    echo "📦 npm: $(npm --version)"
    echo ""
    echo "🎮 控制说明:"
    echo "   - 鼠标左键拖拽: 旋转视角"
    echo "   - 鼠标滚轮: 缩放"
    echo "   - 鼠标右键拖拽: 平移视角"
    echo "   - 点击物体: 查看信息"
    echo ""
    echo "=========================================="
    echo ""
}

# 清理函数
cleanup() {
    log_info "正在清理资源..."
    # 这里可以添加清理逻辑
    log_success "清理完成"
}

# 设置信号处理
trap cleanup EXIT INT TERM

# 主函数
main() {
    log_info "开始启动 Three.js 项目..."
    
    # 执行检查
    check_node_version
    check_npm_version
    check_dependencies
    check_required_files
    set_permissions
    create_log_dir
    
    # 显示启动信息
    show_startup_info
    
    # 启动健康检查
    start_health_check
    
    log_success "Three.js 项目启动完成！"
    
    # 执行传入的命令
    exec "$@"
}

# 如果直接运行此脚本
if [ "${BASH_SOURCE[0]}" == "${0}" ]; then
    main "$@"
fi
