# Python3.12 + Node.js 全栈应用项目

这是一个使用Python Flask和Node.js Express构建的全栈Web应用示例，展示了前后端分离架构的基本实现。

## 项目结构

```
python-node-template/
├── backend/                 # Python后端
│   ├── app.py              # Flask应用主文件
│   └── requirements.txt    # Python依赖
├── frontend/               # Node.js前端
│   ├── server.js           # Express服务器
│   ├── package.json        # Node.js依赖
│   └── public/
│       └── index.html      # 前端页面
├── Dockerfile              # Docker容器化配置
├── start.sh               # 启动脚本
├── Makefile               # 构建和运行命令
└── README.md              # 项目说明文档
```

## 功能特性

- ✅ 基于Python Flask的RESTful API后端
- ✅ 基于Node.js Express的Web前端服务器
- ✅ 支持CORS跨域请求
- ✅ 美观的响应式前端界面
- ✅ 实时API测试功能
- ✅ Makefile自动化构建、运行和管理
- ✅ 支持后台运行和进程管理
- ✅ Docker容器化支持
- ✅ 环境变量配置支持

## 环境要求

- Python 3.11 或更高版本
- Node.js 18 或更高版本
- Make工具（可选，用于自动化构建）
- Docker（可选，用于容器化运行）

## 快速开始

### 方法一：直接运行

```bash
# 安装Python依赖
cd backend && pip install -r requirements.txt

# 安装Node.js依赖
cd ../frontend && npm install

# 启动后端服务（新终端）
cd backend && python app.py

# 启动前端服务（新终端）
cd frontend && node server.js
```

### 方法二：使用Makefile

```bash
# 安装所有依赖
make install

# 启动所有服务（后台运行）
make run

# 分别启动服务
make run-backend  # 启动Python后端 (端口8000)
make run-frontend # 启动Node.js前端 (端口8001)

# 查看服务状态
make status

# 停止所有服务
make stop

# 重启所有服务
make restart

# 开发模式（热重载）
make dev

# 清理日志文件
make clean

# 查看帮助
make help
```

### 方法三：使用Docker

```bash
# 构建Docker镜像
make docker-build

# 运行容器
make docker-run

# 运行容器并指定端口
docker run -p 3000:8001 -p 5000:8000 python-node-app
```

## 应用使用示例

### 启动应用
```bash
make run
# 输出：
# ✅ Python后端已启动，端口: 8000
# ✅ Node.js前端已启动，端口: 8001
# 前端页面: http://localhost:8001
# 后端API: http://localhost:8000
```

### 访问应用
```bash
# 访问前端页面
open http://localhost:8001

# 测试后端API
curl http://localhost:8000/api/hello
# 输出：{"message": "Hello from Python backend!", "status": "success", ...}
```

### 查看服务状态
```bash
make status
# 输出：
# ✅ 后端服务正在运行，端口: 8000
# ✅ 前端服务正在运行，端口: 8001
```

### 停止服务
```bash
make stop
# 输出：
# ✅ 所有服务已停止
```

## 代码说明

### backend/app.py

Flask后端应用包含以下功能：

- **Flask应用初始化**: 创建Flask应用实例并配置CORS
- **API路由**: 实现RESTful API接口
- **端口配置**: 从环境变量BACKEND_PORT读取端口，默认8000
- **JSON响应**: 返回结构化的JSON数据
- **错误处理**: 基本的错误处理机制

### frontend/server.js

Express前端服务器包含以下功能：

- **静态文件服务**: 提供HTML、CSS、JS等静态资源
- **代理功能**: 将API请求代理到后端服务
- **端口配置**: 从环境变量FRONTEND_PORT读取端口，默认8001
- **CORS处理**: 处理跨域请求

### frontend/public/index.html

前端页面包含以下功能：

- **响应式设计**: 适配不同屏幕尺寸
- **API测试**: 提供GET和POST API测试功能
- **实时显示**: 动态显示API响应结果
- **现代UI**: 使用现代CSS样式和交互效果

### Makefile

提供了以下命令：

- `install`: 安装Python和Node.js依赖
- `run`: 启动所有服务（后台运行）
- `run-backend`: 启动Python后端服务
- `run-frontend`: 启动Node.js前端服务
- `stop`: 停止所有服务
- `restart`: 重启所有服务
- `status`: 查看服务运行状态
- `dev`: 开发模式（热重载）
- `clean`: 清理日志文件
- `docker-build`: 构建Docker镜像
- `docker-run`: 运行Docker容器
- `help`: 显示帮助信息

### Dockerfile

- 多阶段构建，包含Python和Node.js环境
- 自动安装依赖和启动服务
- 支持端口配置（环境变量）
- 暴露8000和8001端口

## API接口说明

### GET /api/hello
返回简单的问候信息

**响应示例:**
```json
{
  "message": "Hello from Python backend!",
  "status": "success",
  "data": {
    "backend": "Python Flask",
    "frontend": "Node.js",
    "timestamp": "2024-01-01T00:00:00Z"
  }
}
```

### POST /api/info
接收前端发送的数据并返回处理结果

**请求体:**
```json
{
  "name": "张三",
  "message": "你好，这是来自前端的消息！"
}
```

**响应示例:**
```json
{
  "message": "Hello 张三!",
  "received_message": "你好，这是来自前端的消息！",
  "status": "success",
  "backend_response": "Data received successfully from Python backend"
}
```

## 技术栈

### 后端
- Python 3.11
- Flask 2.3.3
- Flask-CORS 4.0.0
- Gunicorn 21.2.0

### 前端
- Node.js 18
- Express 4.18.2
- Axios 1.6.0
- HTML5 + CSS3 + JavaScript

## 环境变量

- `FRONTEND_PORT`: 前端服务端口 (默认: 8001)
- `BACKEND_PORT`: 后端服务端口 (默认: 8000)
- `BACKEND_URL`: 后端API地址 (默认: http://localhost:8000)

## 扩展建议

1. **数据库集成**: 添加SQLite、PostgreSQL或MongoDB数据库支持
2. **用户认证**: 实现JWT或Session-based用户认证系统
3. **API文档**: 使用Swagger/OpenAPI生成API文档
4. **日志记录**: 添加结构化日志记录和监控
5. **配置管理**: 使用环境变量或配置文件管理应用配置
6. **测试覆盖**: 添加单元测试和集成测试
7. **CI/CD**: 配置GitHub Actions或GitLab CI进行自动化部署
8. **缓存机制**: 添加Redis缓存提升性能
9. **API版本控制**: 实现API版本管理
10. **安全加固**: 添加输入验证、SQL注入防护等安全措施
11. **性能优化**: 添加数据库连接池、请求限流等优化
12. **容器编排**: 使用Docker Compose或Kubernetes进行容器编排

## 故障排除

1. **端口冲突**: 如果端口被占用，可以修改Makefile中的端口配置
2. **依赖安装失败**: 确保Python和Node.js版本符合要求
3. **跨域问题**: 后端已配置CORS，如果仍有问题请检查防火墙设置
4. **服务启动失败**: 检查端口是否被占用，查看日志文件排查问题
5. **Docker构建失败**: 确保Docker服务正在运行，检查网络连接

## 许可证

此项目仅用于学习和演示目的。
