# PHP Hello World 应用

基于 PHP 7.4 的简单 Web 应用模板，使用 PHP 内置服务器提供 Web 服务。

## 项目特性

- 🐘 **PHP 7.4+** - 使用稳定的 PHP 版本
- 🌐 **PHP 内置服务器** - 无需 Apache/Nginx，开箱即用
- 🐳 **Docker 支持** - 基于官方 PHP 镜像的容器化部署
- 🔧 **环境变量配置** - 端口可通过环境变量配置
- 📦 **Composer 依赖管理** - 现代化的 PHP 包管理
- 🧪 **PHPUnit 测试** - 集成测试框架
- 🚀 **一键部署** - 支持 Makefile 和 .eazy 配置

## 快速开始

### 环境要求

- PHP 7.4 或更高版本
- Composer（PHP 包管理器）
- Docker（可选，用于容器化部署）

### 本地开发

1. **克隆项目**
   ```bash
   git clone <项目地址>
   cd php7.4-template
   ```

2. **安装依赖**
   ```bash
   make install
   ```

3. **启动应用**
   ```bash
   make run
   ```

4. **访问应用**
   打开浏览器访问：http://localhost:8000

5. **停止应用**
   ```bash
   make stop
   ```

### 环境变量配置

| 变量名 | 默认值 | 说明 |
|--------|--------|------|
| `APP_PORT_1` | 8000 | 应用监听端口 |

### Docker 部署

1. **构建镜像**
   ```bash
   docker build -t php-hello-world .
   ```

2. **运行容器**
   ```bash
   docker run -p 8000:8000 -e APP_PORT_1=8000 php-hello-world
   ```

## 项目结构

```
php7.4-template/
├── .eazy                    # 应用配置文件
├── .resource/
│   └── resource_dev.json   # 资源配置文件
├── .vscode/                # VS Code 配置
├── .gitignore              # Git 忽略文件
├── hello_world.php         # 主程序
├── composer.json           # PHP 依赖配置
├── Dockerfile              # Docker 构建文件
├── docker-entrypoint.sh    # Docker 启动脚本
├── Makefile               # 构建和运行脚本
└── README.md              # 项目说明文档
```

## 开发指南

### 添加新路由

在 `hello_world.php` 中添加新的处理逻辑：

```php
<?php
// 获取请求路径
$requestUri = $_SERVER['REQUEST_URI'];

switch ($requestUri) {
    case '/':
        // 首页逻辑
        break;
    case '/api/status':
        // API 状态检查
        break;
    default:
        // 404 处理
        http_response_code(404);
        echo json_encode(['error' => 'Not Found']);
        break;
}
?>
```

### 添加依赖

在 `composer.json` 的 `require` 部分添加新的依赖：

```json
{
    "require": {
        "php": ">=7.4.0",
        "monolog/monolog": "2.8.0"
    }
}
```

然后运行：
```bash
composer update
```

### 配置说明

- **composer.json**: PHP 依赖管理配置，包含 PHPUnit 测试框架
- **Dockerfile**: 基于 PHP 7.4-cli 镜像，使用 Composer 管理依赖
- **Makefile**: 提供 install、run、stop 命令
- **.eazy**: 应用部署配置文件

## API 接口

### GET /

返回应用状态信息的 JSON 响应。

**响应示例：**
```json
{
    "message": "Hello, World!",
    "port": "8000",
    "timestamp": "2025-01-07 16:20:00",
    "php_version": "7.4.33",
    "server": "PHP Built-in Server"
}
```

## 测试

### 运行测试

```bash
# 使用 Composer 运行测试
composer test

# 或直接使用 PHPUnit
./vendor/bin/phpunit
```

### 创建测试

在项目根目录创建 `tests/` 目录，添加测试文件：

```php
<?php
// tests/HelloWorldTest.php
use PHPUnit\Framework\TestCase;

class HelloWorldTest extends TestCase
{
    public function testHelloWorld()
    {
        $this->assertEquals('Hello, World!', 'Hello, World!');
    }
}
?>
```

## 部署说明

### 使用 .eazy 配置部署

项目包含 `.eazy` 配置文件，支持一键部署：

- **端口映射**: APP_PORT_1 (8000) 作为主端口
- **资源限制**: CPU 0.5核，内存 2GB，存储 1GB
- **访问入口**: 支持外网域名访问

### 生产环境建议

1. **使用 Web 服务器**: 生产环境建议使用 Nginx + PHP-FPM
2. **性能优化**: 启用 OPcache 和 APCu
3. **安全配置**: 配置 HTTPS 和防火墙规则
4. **监控日志**: 集成日志收集和监控系统
5. **健康检查**: 添加健康检查端点

### Nginx 配置示例

```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /var/www/html;
    index hello_world.php;

    location / {
        try_files $uri $uri/ /hello_world.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass 127.0.0.1:9000;
        fastcgi_index hello_world.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }
}
```

## 故障排除

### 常见问题

1. **端口被占用**
   ```bash
   # 查看端口占用
   lsof -i :8000
   # 或使用 make stop 停止应用
   ```

2. **Composer 依赖安装失败**
   - 检查网络连接
   - 清除 Composer 缓存：`composer clear-cache`
   - 更新 Composer：`composer self-update`

3. **Docker 构建失败**
   - 检查 Dockerfile 语法
   - 确认基础镜像可用
   - 检查网络连接

4. **PHP 版本不兼容**
   - 确保使用 PHP 7.4 或更高版本
   - 检查 `composer.json` 中的 PHP 版本要求

## 性能优化

### PHP 配置优化

在 `php.ini` 中启用以下配置：

```ini
; 启用 OPcache
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=4000

; 启用 APCu
apc.enabled=1
apc.shm_size=128M
```

### 应用优化

1. **使用 Composer 自动加载优化**
   ```bash
   composer dump-autoload --optimize
   ```

2. **启用 Composer 类映射优化**
   ```json
   {
       "config": {
           "optimize-autoloader": true
       }
   }
   ```

## 贡献指南

1. Fork 项目
2. 创建特性分支 (`git checkout -b feature/AmazingFeature`)
3. 提交更改 (`git commit -m 'Add some AmazingFeature'`)
4. 推送到分支 (`git push origin feature/AmazingFeature`)
5. 打开 Pull Request

## 许可证

本项目采用 MIT 许可证 - 查看 [LICENSE](LICENSE) 文件了解详情。

## 联系方式

如有问题或建议，请通过以下方式联系：

- 项目 Issues: [GitLab Issues](项目地址/-/issues)
- 邮箱: [联系邮箱]

---

**Happy Coding with PHP! 🐘**
