#!/bin/bash

# KaboomJS游戏Docker启动脚本
# 此脚本用于在容器启动时进行必要的初始化工作

set -e

# 颜色输出函数
print_info() {
    echo -e "\033[36m[INFO]\033[0m $1"
}

print_success() {
    echo -e "\033[32m[SUCCESS]\033[0m $1"
}

print_error() {
    echo -e "\033[31m[ERROR]\033[0m $1"
}

print_warning() {
    echo -e "\033[33m[WARNING]\033[0m $1"
}

# 显示启动信息
print_info "=== KaboomJS 游戏容器启动 ==="
print_info "Node.js 版本: $(node --version)"
print_info "NPM 版本: $(npm --version)"
print_info "工作目录: $(pwd)"
print_info "用户: $(whoami)"

# 检查必要文件
print_info "检查必要文件..."
required_files=("package.json" "index.html" "game.js")

for file in "${required_files[@]}"; do
    if [ -f "$file" ]; then
        print_success "✓ $file 存在"
    else
        print_error "✗ $file 不存在"
        exit 1
    fi
done

# 检查node_modules
if [ ! -d "node_modules" ]; then
    print_warning "node_modules 不存在，正在安装依赖..."
    npm install
    print_success "依赖安装完成"
else
    print_success "✓ node_modules 存在"
fi

# 检查端口环境变量
if [ -z "$APP_PORT_1" ]; then
    export APP_PORT_1=8000
    print_warning "APP_PORT_1 环境变量未设置，使用默认端口: $APP_PORT_1"
else
    print_info "使用端口: $APP_PORT_1"
fi
export PORT=$APP_PORT_1

# 创建必要的目录
print_info "创建必要目录..."
mkdir -p logs
mkdir -p tmp

# 设置权限
print_info "设置文件权限..."
chmod -R 755 .

# 显示游戏信息
print_info "=== 游戏信息 ==="
if [ -f "package.json" ]; then
    GAME_NAME=$(node -p "require('./package.json').name" 2>/dev/null || echo "KaboomJS Game")
    GAME_VERSION=$(node -p "require('./package.json').version" 2>/dev/null || echo "1.0.0")
    print_info "游戏名称: $GAME_NAME"
    print_info "游戏版本: $GAME_VERSION"
fi

# 显示访问信息
print_info "=== 访问信息 ==="
print_info "本地访问: http://localhost:$APP_PORT_1"
print_info "容器内访问: http://0.0.0.0:$APP_PORT_1"

# 健康检查函数
health_check() {
    print_info "执行健康检查..."
    if curl -f -s "http://localhost:$PORT" > /dev/null; then
        print_success "健康检查通过"
        return 0
    else
        print_error "健康检查失败"
        return 1
    fi
}

# 优雅关闭处理
cleanup() {
    print_info "收到关闭信号，正在优雅关闭..."
    if [ ! -z "$SERVER_PID" ]; then
        kill -TERM "$SERVER_PID" 2>/dev/null || true
        wait "$SERVER_PID" 2>/dev/null || true
    fi
    print_success "应用已关闭"
    exit 0
}

# 设置信号处理
trap cleanup SIGTERM SIGINT

# 如果传入了参数，直接执行
if [ "$#" -gt 0 ]; then
    print_info "执行命令: $@"
    exec "$@"
fi

# 默认启动游戏服务器
print_success "=== 启动 KaboomJS 游戏服务器 ==="
print_info "服务器将在 http://0.0.0.0:$APP_PORT_1 上运行"
print_info "按 Ctrl+C 停止服务器"

# 启动服务器并获取PID
npm start &
SERVER_PID=$!

# 等待服务器启动
sleep 3

# 执行健康检查
if health_check; then
    print_success "🎮 KaboomJS 游戏服务器启动成功！"
    print_info "现在可以在浏览器中访问游戏了"
else
    print_warning "健康检查失败，但服务器可能仍在启动中..."
fi

# 等待服务器进程
wait "$SERVER_PID"
